# btc16m_full.py
# Fully saturated 64-seed vector HDGL covenant
# Logarithmic folding to 16,777,216 vectors

MAX_SEED_VECTORS = 64
TARGET_VECTORS = 16_777_216  # 2^24
BLOCK_SIZE = 4
SAFE_ZERO = "0x0"

# Fully saturated r_dim & Ω for 64 vectors
# Powers-of-2 scaled for safety, unique for each block
PARAMS = {i+1: (f"0x{hex(0x3000*(i+1))[2:].upper()}", f"0x{hex(0x812//(i+1) if i else 0x812)[2:].upper()}") 
          for i in range(MAX_SEED_VECTORS // BLOCK_SIZE)}

def generate_hdgl_covenant(pubkey="<YOUR_PUBLIC_KEY>"):
    script = [f"{pubkey} OP_CHECKSIGVERIFY\n"]

    # 1️⃣ Generate 64 fully saturated seed vectors
    num_blocks = MAX_SEED_VECTORS // BLOCK_SIZE
    for block_idx in range(1, num_blocks + 1):
        # vector addition
        v_line = " ".join([SAFE_ZERO]*BLOCK_SIZE) + " OP_HDGL_VEC_ADD OP_EQUALVERIFY"
        script.append(v_line)

        # OP_ADD + OP_EQUALVERIFY
        idx_base = 100 + (block_idx-1)*BLOCK_SIZE
        add_line = f"{idx_base} {idx_base+1} OP_ADD {idx_base+2} OP_EQUALVERIFY"
        script.append(add_line)

        # OP_WITHINVERIFY
        within_line = f"{idx_base+3} {idx_base+4} {idx_base+5} OP_WITHINVERIFY"
        script.append(within_line)

        # embed fully unique r_dim & Ω
        param_line = f"{PARAMS[block_idx][0]} {PARAMS[block_idx][1]} OP_HDGL_PARAM_ADD OP_EQUALVERIFY"
        script.append(param_line)
        script.append("")  # spacing

    # 2️⃣ Logarithmic folding to reach TARGET_VECTORS
    seed_count = MAX_SEED_VECTORS
    fold_index = 0
    while seed_count < TARGET_VECTORS:
        fold_index += 1
        prev_count = seed_count
        seed_count *= 2

        v_line = f"OP_VEC_DUP {prev_count} OP_HDGL_VEC_ADD OP_EQUALVERIFY"
        script.append(v_line)

        idx_base = 100 + prev_count  # safe conservative indexing
        add_line = f"{idx_base} {idx_base+1} OP_ADD {idx_base+2} OP_EQUALVERIFY"
        script.append(add_line)
        within_line = f"{idx_base+3} {idx_base+4} {idx_base+5} OP_WITHINVERIFY"
        script.append(within_line)
        script.append("")  # spacing

    script.append("OP_CHECKTEMPLATEVERIFY")
    return "\n".join(script)

if __name__ == "__main__":
    covenant_script = generate_hdgl_covenant()
    print("=== Fully Saturated 64-Seed Vector HDGL Covenant Folded to 16,777,216 Vectors ===")
    print(covenant_script)
